package Jellybean::Handler::Maypole;

use strict;
use warnings;

use Carp;
use Class::Roles does => 'Jellybean::Handler';
use UNIVERSAL::require;

use vars '$VERSION';
$VERSION = '0.12';

sub new
{
	my ($class, %args) = @_;
	my $app            = $args{app};
	croak( 'No Maypole app class provided' ) unless $app;

	eval { $app->require() } or 
		croak( "Couldn't require '$app': $@\n" );

	Class::Roles->import(
		apply => 
		{
			role => 'Jellybean::Handler::Roles::Maypole',
			to   => $app,
		},
	);

	bless \%args, $class;
}

sub formatter_type {}

sub app
{
	my $self = shift;
	return $self->{app};
}

sub activate
{
	my ($self, $request, $response) = @_;
	my $app                         = $self->app();

	# this config can only last for the duration of this request
	my $config               = $app->config();

	$config->{template_root} = $self->{templates};
	$config->{uri_base}      = $self->{uri_base};

	$config->{jb} =
	{
		_request  => $request,
		_response => $response,
	};

	$app->handler();
	delete $config->{jb};
}

package Jellybean::Handler::Roles::Maypole;

use Class::Roles role => [qw( send_output parse_location get_request )];

sub get_request { return }

sub send_output
{
	my $self = shift;
	$self->config->{jb}{_response}->body( $self->{output} );
}

sub parse_location
{
	my $self        = shift;
	my $request     = $self->config->{jb}{_request};

	$self->{path}   = join('/', $request->path() );

	$self->{params} = $request->params();
	$self->{query}  = $request->params();

	$self->parse_path();
}

1;
__END__

=head1 NAME

Jellybean::Handler::Maypole - a Jellybean handler for Maypole applications

=head1 SYNOPSIS

	# prevent Maypole from loading mod_perl unnecessarily
	BEGIN
	{
		$INC{'Apache.pm'}         = 1;
		$INC{'Apache/Request.pm'} = 1;
	}

	# launch Jellybean as normal
	# create this handler with:
	use Jellybean::Handler::Maypole;

	my $jcm = Jellybean::Handler::Maypole->new(
		app       => 'Your::Maypole::App',
		templates => 'path/to/app/templates',
		uri_base  => 'http://localhost:port/path/',
	);

	# add the handler to a Jellybean::Engine instance

=head1 DESCRIPTION

Jellybean::Handler::Maypole runs Maypole applications through Jellybean.
Note that this currently only works with Jellybean's HTTP transport.  In
effect, you can run Maypole applications without mod_perl.  I recommend using a
SQLite database, as then everything is easy to install.

Since this is early software and the current version of Maypole doesn't know
much about running I<without> mod_perl, things may not work perfectly.  They
should be pretty close though.

B<Note:> If you want to serve other files, such as CSS, be sure to install a
L<Jellybean::Handler::File> handler as the default handler.  You I<probably>
don't want your Maypole handler as the default handler, lest it try to serve a
file and become very confused.

=head1 METHODS

There are only three methods you need to know about.

=over 4

=item new( %args )

Creates and returns a new handler.  Pass in three arguments as key-value pairs:

=over

=item C<app>

The name of the module that implements your Maypole application.

=item C<template_root>

The path to the template files for your Maypole application.

=item C<uri_base>

The URI at which you can request your Maypole application.  Note that you'll
probably want a File Handler as your default handler and this handler bound to
a different path component.

You I<can> have multiple Maypole Handlers listening on the same server.

=back

=item app()

Returns the name of the Maypole application module this handler serves.

=item activate( $request, $response )

Activates the Maypole application due to an incoming Jellybean request.

=back

This module also provides the Jellybean::Handler::Roles::Maypole role (see
L<Class::Roles>), but does so behind the scenes where you'll probably never
need to know it exists.

=head1 AUTHOR

chromatic, E<lt>chromatic at wgz dot orgE<gt>

=head1 BUGS

No known bugs.

=head1 COPYRIGHT

Copyright (c) 2004, chromatic.  All rights reserved.

This module is free software; you can use, redistribute, and modify it under
the same terms as Perl itself.
